class EasyPage < ActiveRecord::Base
  include Redmine::SafeAttributes

  CUSTOM_PAGE = 'easy-custom'
  IDENTIFIER_MAX_LENGTH = 100
  PAGE_LAYOUTS = {
    'tchtrrs' => {
      :path => 'easy_page_layouts/two_column_header_three_rows_right_sidebar',
      :zones => ['top-left', 'middle-left', 'middle-right', 'bottom-left', 'right-sidebar']},
    'tchfw' => {
      :path => 'easy_page_layouts/two_column_header_first_wider',
      :zones => ['top-middle', 'middle-left', 'middle-right']},
    'fdd' => {
      :path => 'easy_page_layouts/full_dynamic_dashboard',
      :zones => ['top-middle']},
    'tchaf' => {
      :path => 'easy_page_layouts/two_column_header_and_footer',
      :zones => ['top-middle', 'middle-left', 'middle-right', 'middle2-left', 'middle2-right', 'middle3-left', 'middle3-right', 'mini-middle3-1', 'mini-middle3-2', 'mini-middle3-3', 'mini-middle4-1', 'mini-middle4-2', 'mini-middle4-3', 'mini-middle4-4', 'mini-middle5-1', 'mini-middle5-2', 'mini-middle5-3', 'mini-middle5-4', 'mini-middle5-5', 'bottom-middle']}
  }

  belongs_to :user

  has_many :zones, lambda { preload(:zone_definition).order("#{EasyPageAvailableZone.table_name}.position ASC") }, :class_name => 'EasyPageAvailableZone', :foreign_key => 'easy_pages_id', :dependent => :destroy
  has_many :modules, lambda { preload(:module_definition) }, :class_name => 'EasyPageAvailableModule', :foreign_key => 'easy_pages_id', :dependent => :destroy
  has_many :templates, :class_name => 'EasyPageTemplate', :foreign_key => 'easy_pages_id', :dependent => :destroy

  has_many :all_modules, :through => :zones
  has_many :easy_page_tabs, :class_name => 'EasyPageUserTab', :foreign_key => 'page_id', :dependent => :destroy

  scope :for_index, lambda { where(["#{EasyPage.table_name}.has_template = ? OR #{EasyPage.table_name}.is_user_defined = ?", true, true]) }

  acts_as_taggable_on :tags
  EasyExtensions::EasyTag.register self, {:easy_query_class => 'EasyPageQuery', :referenced_collection_name => 'easy_pages'}

  attr_protected :id

  validates_length_of :page_name, :in => 1..50, :allow_nil => false
  validates_uniqueness_of :identifier, :allow_blank => true, :scope => [:entity_type, :entity_id, :user_id], :if => Proc.new { |p| EasyPage.column_names.include?('identifier') }
  validates_length_of :identifier, :allow_blank => true, :in => 1..IDENTIFIER_MAX_LENGTH, :if => Proc.new { |p| EasyPage.column_names.include?('identifier') }
  validates_format_of :identifier, :with => /\A(?!\d+$)[a-z0-9\-_]*\z/, :if => Proc.new { |p| EasyPage.column_names.include?('identifier') && p.identifier_changed? }
  validates_exclusion_of :identifier, :in => %w( index show new create edit update destroy templates), :if => Proc.new { |p| EasyPage.column_names.include?('identifier') }
  validates :layout_path, inclusion: PAGE_LAYOUTS.collect{|_, v| v[:path]}

  before_save :change_page_name
  after_save :ensure_available_page_zones

  safe_attributes 'identifier', 'user_defined_name', 'tag_list'

  @easy_pages = {}

  def self.find_similiar(page_name)
    EasyPage.where(["#{EasyPage.table_name}.page_name LIKE ?", "#{page_name}-%"])
  end

  def user_modules(user = nil, entity_id = nil, tab = 1, options={})
    tab = tab.to_i
    tab = 1 if tab <= 0

    user_tab_modules(tab, user, entity_id, options)
  end

  def user_tab_modules(tab, user = nil, entity_id = nil, options={})
    scope = EasyPageZoneModule.preload(:zone_definition, :module_definition).joins(:available_zone).readonly(false).order("#{EasyPageAvailableZone.table_name}.position ASC", "#{EasyPageZoneModule.table_name}.position ASC")

    user = User.find(user) if (!user.nil? && !user.is_a?(User))
    scope = scope.where(user_id: user, entity_id: entity_id, easy_pages_id: self)
    scope = scope.where(tab_id: tab) unless options[:all_tabs]

    all_modules = scope.reject{|mod| Redmine::Plugin.disabled?(mod.module_definition.try(:registered_in_plugin)) }

    if options[:without_zones]
      all_modules
    else
      page_modules = self.zones.joins(:zone_definition).pluck(:zone_name).each_with_object({}) { |zone_name, result| result[zone_name] = [] }
      page_modules.merge!(all_modules.group_by{|mod| mod.zone_definition.zone_name })
      page_modules
    end
  end

  def translated_name
    l("easy_pages.pages.#{page_name.underscore}", :default => user_defined_name)
  end

  def translated_description
    l("easy_pages.pages_description.#{page_name.underscore}")
  end

  def unassigned_zones
    assigned_zones = self.zones.collect{|zone| zone.easy_page_zones_id}
    assigned_zones ||= []

    scope = EasyPageZone.all
    scope = scope.where("#{EasyPageZone.table_name}.id NOT IN (#{assigned_zones.join(',')})") if assigned_zones.size > 0

    scope.to_a
  end

  def available_modules
    self.modules.select do |m|
      m && m.module_definition && !Redmine::Plugin.disabled?(m.module_definition.registered_in_plugin)
    end
  end

  def ensure_available_page_zones
    plz = PAGE_LAYOUTS.detect{|_, p| p[:path] == self.layout_path}
    return true if plz.nil?

    page_layout_zones = plz[1][:zones]

    page_layout_zones.each do |page_layout_zone|
      EasyPageAvailableZone.ensure_easy_page_available_zone self, page_layout_zone
    end

    return true
  end

  def install_registered_modules
    EasyPageModule.install_all_registered_modules_to_page self
  end

  def default_template
    self.templates.where(:is_default => true).first
  end

  private

  def change_page_name
    self.page_name = self.page_name.gsub(/[ ]/, '-').dasherize unless self.page_name.nil?
  end

end
