require File.expand_path('../../spec_helper', __FILE__)

describe TimeEntry do
  let(:user){ FactoryGirl.create(:user) }
  let!(:project){ FactoryGirl.create(:project, number_of_issues: 1, members: [user]) }
  let(:easy_global_time_entry_setting) { FactoryGirl.create(:easy_global_time_entry_setting, role_id: user.roles.first.id) }
  let(:time_entry_current) { FactoryGirl.build(:time_entry, :current, user_id: user.id, project_id: project.id, issue_id: project.issues.first.id) }
  let(:time_entry_current_ten_days_new) { FactoryGirl.build(:time_entry, :current, spent_on: Date.today + 10.day) }
  let(:time_entry_old) { FactoryGirl.build(:time_entry, :old, user_id: user.id, project_id: project.id, issue_id: project.issues.first.id) }
  let(:time_entry_future) { FactoryGirl.build(:time_entry, :future, user_id: user.id, project_id: project.id, issue_id: project.issues.first.id) }
  let(:locked_time_entry) { FactoryGirl.build(:time_entry, user_id: user.id, project_id: project.id, issue_id: project.issues.first.id, easy_locked: true) }

  it 'respects limits' do
    with_current_user(user) do
      expect( user.roles_for_project(project).empty?).to eq(false)
      expect( easy_global_time_entry_setting.valid?).to eq(true)
      expect( time_entry_current.valid? ).to eq(true)
      expect( time_entry_current_ten_days_new.valid? ).to eq(true)
      expect( time_entry_old.valid? ).to eq(false)
      expect( time_entry_future.valid? ).to eq(false)
    end
  end

  it 'respects create limits' do
    with_current_user(user) do
      expect( user.roles_for_project(project).empty?).to eq(false)

      easy_global_time_entry_setting.spent_on_limit_before_today = 2
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = 10
      easy_global_time_entry_setting.save

      expect( time_entry_old.valid? ).to eq(false)

      easy_global_time_entry_setting.spent_on_limit_before_today = 10
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = 10
      easy_global_time_entry_setting.save

      expect( time_entry_old.valid? ).to eq(true)
    end
  end

  it 'respects edit limits' do
    with_current_user(user) do
      expect( user.roles_for_project(project).empty?).to eq(false)
      time_entry_old.save(:validate => false)

      easy_global_time_entry_setting.spent_on_limit_before_today = 10
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = 2
      easy_global_time_entry_setting.save

      expect( time_entry_old.valid? ).to eq(false)

      easy_global_time_entry_setting.spent_on_limit_before_today = 10
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = 10
      easy_global_time_entry_setting.save

      expect( time_entry_old.valid? ).to eq(true)
    end
  end

  it 'respects no limits' do
    with_current_user(user) do
      expect( user.roles_for_project(project).empty?).to eq(false)

      easy_global_time_entry_setting.spent_on_limit_before_today = nil
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = 2
      easy_global_time_entry_setting.save
      expect( time_entry_old.valid? ).to eq(true)

      time_entry_old.save(:validate => false)

      easy_global_time_entry_setting.spent_on_limit_before_today = 2
      easy_global_time_entry_setting.spent_on_limit_before_today_edit = nil
      easy_global_time_entry_setting.save

      expect( time_entry_old.valid? ).to eq(true)
    end
  end

  it 'can remove project with locked time entry' do
    locked_time_entry.save
    project.reload
    expect{ project.destroy }.to change(Project, :count).by(-1)
  end

  it 'can validate time_entry without spent_on field', :logged => true do
    time_entry_current.spent_on = nil
    expect(time_entry_current.should_validate_time_entry_for_workers?).to eq(true)
    expect(time_entry_current.valid?).to eq(false)
  end

end
