require File.expand_path('../../spec_helper', __FILE__)

describe CustomField do

  context 'value tree' do
    context 'before save' do
      it 'simple' do
        cf = CustomField.new(name: 'value_tree_1', field_format: 'value_tree')
        cf.possible_values = [
          'Value 1',
          '> Value 1.1',
          '> Value 1.2',
          'Value 2'
        ]

        expect(cf.save).to be(true)
        expect(cf.possible_values).to eq([
          'Value 1',
          'Value 1 > Value 1.1',
          'Value 1 > Value 1.2',
          'Value 2'
        ])
      end

      it 'complex' do
        cf = CustomField.new(name: 'value_tree_2', field_format: 'value_tree')
        cf.possible_values = [
          'Value 1',
          '> Value 1.1',
          '> Value 1.2',
          '>> Value 1.2.1',
          '>>> Value 1.2.1.1',
          '>> Value 1.2.2',
          '>> Value 1.2.3',
          'Value 2',
          '> Value 2.1',
          '>> Value 2.1.1',
          '>>> Value 2.1.1.1',
          '>>>> Value 2.1.1.1.1',
          'Value 3'
        ]

        expect(cf.save).to be(true)
        expect(cf.possible_values).to eq([
          'Value 1',
          'Value 1 > Value 1.1',
          'Value 1 > Value 1.2',
          'Value 1 > Value 1.2 > Value 1.2.1',
          'Value 1 > Value 1.2 > Value 1.2.1 > Value 1.2.1.1',
          'Value 1 > Value 1.2 > Value 1.2.2',
          'Value 1 > Value 1.2 > Value 1.2.3',
          'Value 2',
          'Value 2 > Value 2.1',
          'Value 2 > Value 2.1 > Value 2.1.1',
          'Value 2 > Value 2.1 > Value 2.1.1 > Value 2.1.1.1',
          'Value 2 > Value 2.1 > Value 2.1.1 > Value 2.1.1.1 > Value 2.1.1.1.1',
          'Value 3'
        ])
      end

      it 'ended on high level' do
        cf = CustomField.new(name: 'value_tree_3', field_format: 'value_tree')
        cf.possible_values = [
          'Value 1',
          '> Value 1.1',
          '>> Value 1.1.1',
          '>>> Value 1.1.1.1',
          '>>>> Value 1.1.1.1.1',
        ]

        expect(cf.save).to be(true)
        expect(cf.possible_values).to eq([
          'Value 1',
          'Value 1 > Value 1.1',
          'Value 1 > Value 1.1 > Value 1.1.1',
          'Value 1 > Value 1.1 > Value 1.1.1 > Value 1.1.1.1',
          'Value 1 > Value 1.1 > Value 1.1.1 > Value 1.1.1.1 > Value 1.1.1.1.1',
        ])
      end

      it 'validation' do
        cf = CustomField.new(name: 'value_tree_5', field_format: 'value_tree')
        cf.possible_values = [
          '> Value 1',
          'Value 1.1',
        ]

        expect(cf.save).to be(false)

        cf.possible_values = [
          'Value 1',
          '>> Value 1.1',
        ]

        expect(cf.save).to be(false)

        cf.possible_values = [
          'Value 1',
          '> Value 1.1',
          '>> Value 1.1.1',
          '>>>> Value 1.1.1',
        ]

        expect(cf.save).to be(false)

        cf.possible_values = [
          'Value 1',
          '> Value 1.1',
          '>> Value 1.1.1',
          '>>> Value 1.1.1',
          'Value 2',
        ]

        expect(cf.save).to be(true)
      end
    end

    it 'custom field edit' do
      values = [
        'Value 1',
        '> Value 1.1',
        '> Value 1.2',
        'Value 2'
      ]
      saved = [
        'Value 1',
        'Value 1 > Value 1.1',
        'Value 1 > Value 1.2',
        'Value 2'
      ]

      cf = CustomField.new(name: 'value_tree_4', field_format: 'value_tree')
      cf.possible_values = values

      expect(cf.save).to be(true)
      expect(cf.possible_values).to eq(saved)
      expect(cf.format.possible_values_for_edit_page(cf)).to eq(values)
    end
  end

  describe 'generated lists' do
    let(:names) { ["C", "A", "B"] }

    context 'when list of users', logged: :admin do
      let!(:users) { names.collect {|name| FactoryGirl.create(:user, :firstname => name)} }
      let(:user_cf) { FactoryGirl.build(:custom_field, :field_format => 'user') }
      let(:users_names) { user_cf.possible_values_options.collect {|user| user[0]} }
      let(:member) { users.first }
      let(:project) { FactoryGirl.create(:project, :members => [member]) }

      it 'lists users sorted by first name' do
        expect(users_names).to eq users_names.sort
        expect(users_names).not_to be_empty
      end

      it 'lists project members' do
        expect(user_cf.possible_values_options(project).map {|user| user[0]}).to eq [member.name]
      end
    end

    context 'when list of milestones (Version)' do
      let(:issue) { FactoryGirl.create(:issue) }
      let!(:versions) { names.collect {|name| FactoryGirl.create(:version, :project => issue.project, :name => name) } }
      let(:milestone_cf) { FactoryGirl.create(:custom_field, :field_format => 'version') }
      let(:milestones_cf_name_values) { milestone_cf.possible_values_options(issue).collect {|milestone| milestone[0]} }
      let(:milestones_sorted) { Version.all.sort.collect(&:name) }

      it 'lists milestones sorted by Version.<=>' do
        expect(milestones_cf_name_values).to eq milestones_sorted
        expect(milestones_cf_name_values).not_to be_empty
      end
    end
  end

  describe 'formatted custom value' do
    context 'when field format is "file"' do
      context 'when empty value' do
        let(:custom_field) { FactoryGirl.create(:issue_custom_field, :field_format => 'file') }

        it 'is empty value', :x => true do
          expect(custom_field.format.formatted_value(nil, nil, '')).to be_empty
        end
      end
    end
  end
end
