require File.expand_path('../../spec_helper', __FILE__)

feature 'easy project query view', :js => true, :logged => :admin do

  def open_settings
    page.find('#easy-query-toggle-button-settings').click
    wait_for_ajax
  end

  def select_group_by(group_name)
    page.find('#group_by .easy-multiselect-tag-container a.icon-add').click
    page.find("#group_by option[value='#{group_name}']").select_option
  end

  def apply_settings
    page.find('#filter_buttons .apply-link').click
    wait_for_ajax
  end

  let!(:projects) { FactoryGirl.create_list(:project, 3) }
  let(:project_cf) { FactoryGirl.create(:project_custom_field, :field_format => 'string') }
  let(:project) { FactoryGirl.create(:project, :custom_field_values => {"#{project_cf.id}" => 'test_cf'}) }

  scenario 'group by author' do
    visit projects_path
    wait_for_ajax
    open_settings
    select_group_by('author')
    apply_settings
    expect(page.find('.list.projects')).to have_content(projects.first.author.to_s)
  end

  scenario 'group by cf' do
    project
    visit settings_project_path(projects.first)
    cv = 'test group'
    page.find("#project_custom_field_values_#{project_cf.id}_#{projects.first.id}").set(cv)
    page.find('#save-project-info').click

    visit projects_path
    wait_for_ajax
    open_settings
    select_group_by("cf_#{project_cf.id}")
    apply_settings
    expect(page.find('.list.projects')).to have_content(cv)
  end

  scenario 'show expanders' do
    projects.last.update_attribute(:parent_id, projects.first.id)
    # default
    visit projects_path
    expect(page.find('table.list.projects')).to have_css('.project-parent-expander', :count => 1)
    # all columns
    visit projects_path(:set_filter => '1', :easy_query => {:columns_to_export => 'all'})
    expect(page.find('table.list.projects')).to have_css('.project-parent-expander', :count => 1)
  end

  feature 'query filter' do
    let(:default_filter) do
      {
        'easy_project_query_default_filters' => { :is_closed => { :operator => '=', :values => [0] }   }
      }
    end

    context 'when default filters are set' do
      context 'when not changed by user' do
        scenario 'should have label "default"' do
          expect_default_filter
        end
      end

      context 'when changed by user' do
        scenario 'should have label "active"' do
          with_easy_settings(default_filter) do
            visit projects_path(:set_filter => 1, :is_closed => '=0', :favorited => '=1')

            expect(page).to have_css('div#easy-query-toggle-button-filters span.active-filter')
          end
        end
      end

      context 'when unset to no filters' do
        scenario 'should have no label' do
          with_easy_settings(default_filter) do
            visit projects_path(:set_filter => 1)

            expect(page).not_to have_css('div#easy-query-toggle-button-filters span')
          end
        end
      end
    end

    context 'when no default filters and filter not changed by user' do
      scenario 'should have label "default"' do
        expect_default_filter
      end
    end

    def expect_default_filter
      with_easy_settings(default_filter) do
        visit projects_path

        expect(page).to have_css('div#easy-query-toggle-button-filters span.default-filter')
      end
    end
  end
end

feature 'easy project query view', :js => true, :logged => :admin do

  let(:project_cf) { FactoryGirl.create(:project_custom_field, :id => 999, :field_format => 'string') }
  let(:project) { FactoryGirl.create(:project, :custom_field_values => {"#{project_cf.id}" => 'test_cf'}) }

  cf_name = 'cf_999'
  let(:default_filter) do
    {
        'easy_project_query_default_sorting_array' => [[cf_name, 'asc']]
    }
  end

  scenario 'order by cf' do
    project
    with_easy_settings(default_filter) do
      visit projects_path
      expect(page).to have_content(I18n.t(:label_project_index))
    end
  end
end
