module EasyPatch
  module RolePatch

    def self.included(base)
      base.extend(ClassMethods)
      base.include InstanceMethods
      base.include Redmine::SafeAttributes # redmine 3.4

      base.class_eval do

        acts_as_easy_translate

        safe_attributes 'name',
          'assignable',
          'position',
          'issues_visibility',
          'users_visibility',
          'time_entries_visibility',
          'all_roles_managed',
          'managed_role_ids',
          'permissions',
          'permissions_all_trackers',
          'permissions_tracker_ids' # redmine 3.4

        has_and_belongs_to_many :activities, :join_table => 'projects_activity_roles', :class_name => 'TimeEntryActivity', :association_foreign_key => 'activity_id'
        has_and_belongs_to_many :projects, :join_table => 'projects_activity_roles'

        has_many :project_activity_roles, :class_name => 'ProjectActivityRole', :dependent => :delete_all
        has_many :projects, :through => :project_activity_roles
        has_many :role_activities, :through => :project_activity_roles
        has_many :easy_default_query_mappings
        has_one :easy_global_time_entry_setting

        has_and_belongs_to_many :easy_queries, :join_table => "#{table_name_prefix}easy_queries_roles#{table_name_suffix}", :foreign_key => 'role_id'

        # column description doesn't exist when migrating from redmine to easyredmine, see migration 062_insert_builtin_roles.rb
        if column_names.include? 'description'
          validates_length_of :description, :maximum => 255
        end

        after_save :invalidate_cache, :if => :builtin?
        after_destroy :invalidate_cache, :if => :builtin?

        alias_method_chain :consider_workflow?, :easy_extensions

        class << self

          alias_method_chain :find_or_create_system_role, :easy_extensions

        end

        def invalidate_cache
          RequestStore.store[:system_roles] = {}
        end

      end
    end

    module InstanceMethods

      def consider_workflow_with_easy_extensions?
        consider_workflow_without_easy_extensions? || has_permission?(:edit_own_issue) || has_permission?(:edit_assigned_issue)
      end

    end

    module ClassMethods

      def find_or_create_system_role_with_easy_extensions(buildin, name)
        RequestStore.store[:system_roles] ||= {}
        RequestStore.store[:system_roles][buildin] ||= find_or_create_system_role_without_easy_extensions(buildin, name)
      end

    end

  end

end
EasyExtensions::PatchManager.register_model_patch 'Role', 'EasyPatch::RolePatch'
