require File.expand_path('../../spec_helper', __FILE__)

describe EasyQueriesController do

  let(:project){ FactoryGirl.create(:project) }

  describe '#new', logged: :admin do

    it 'set query for all projects without project_id' do
      get :new, type: 'EasyIssueQuery'
      expect( assigns(:easy_query).project ).to be_nil
      expect( assigns(:easy_query) ).to be_is_for_all
    end

    it 'set query for project with project_id' do
      get :new, type: 'EasyIssueQuery', project_id: project
      expect( assigns(:easy_query).project ).to eq(project)
      expect( assigns(:easy_query) ).not_to be_is_for_all
    end

    it 'set right filters and group_by' do
      get :new, type: 'EasyIssueQuery', status_id: "o|7", author_id: "=|me", group_by: 'project'
      expect( assigns(:easy_query).filters.keys ).to include('status_id', 'author_id')
      expect( assigns(:easy_query).filters['status_id'][:operator] ).to eq('o')
      expect( assigns(:easy_query).group_by ).to eq(['project'])
    end

  end

  describe '#create', logged: :admin do
    it 'set query attributes from params' do
      post :create, type: 'EasyIssueQuery', easy_query: {name: 'TestName', visibility: EasyQuery::VISIBILITY_PRIVATE},
                    project_id: project, status_id: "o|7", author_id: "=|me", group_by: 'author', confirm: '1'
      expect( assigns(:easy_query) ).not_to be_new_record
      expect( assigns(:easy_query) ).not_to be_is_for_all
      expect( assigns(:easy_query).name ).to eq('TestName')
      expect( assigns(:easy_query).project ).to eq(project)
      expect( assigns(:easy_query).filters.keys ).to include('status_id', 'author_id')
      expect( assigns(:easy_query).filters['status_id'][:operator] ).to eq('o')
      expect( assigns(:easy_query).group_by ).to eq(['author'])
    end

    it 'set multiple grouping for query from params' do
      post :create, type: 'EasyIssueQuery', easy_query: {name: 'TestName', visibility: EasyQuery::VISIBILITY_PRIVATE},
                    project_id: project, group_by: ['author', 'assigned_to'], confirm: '1'
      expect( assigns(:easy_query) ).not_to be_new_record
      expect( assigns(:easy_query) ).not_to be_is_for_all
      expect( assigns(:easy_query).name ).to eq('TestName')
      expect( assigns(:easy_query).project ).to eq(project)
      expect( assigns(:easy_query).group_by ).to eq(['author', 'assigned_to'])
    end
  end

  describe '#update', logged: :admin do
    let(:easy_query){ FactoryGirl.create(:easy_issue_query) }

    it 'set query attributes' do
      patch :update, type: 'EasyIssueQuery', id: easy_query, easy_query: {name: 'TestName'},
                    status_id: "o|7", author_id: "=|me", group_by: 'author', set_filter: '1'
      easy_query.reload
      expect( easy_query.name ).to eq( 'TestName' )
      expect( easy_query.filters.keys ).to include('status_id', 'author_id')
      expect( easy_query.group_by ).to eq( ['author'] )
    end
  end

  describe '#chart', logged: :admin do
    render_views

    def chart_values
      json[:data]['json'].collect{|d| d['values'].to_f}
    end

    def chart_groups
      json[:data]['json'].collect{|d| d['name']}
    end

    let(:page_module) { FactoryGirl.create :easy_page_zone_module, :with_chart_settings }
    let(:page_module_with_incorrect_settings) {
      pm = FactoryGirl.create :easy_page_zone_module, :with_chart_settings
      pm.settings['chart_settings']['axis_y_column'] = 'non_existing_column'
      pm.save
      pm
    }
    let(:chart_request_params) {
      {
       "group_by" => "",
       "load_groups_opened" => "1",
       "set_filter"   => "1",
       "show_avatars" =>"0",
       "show_sum_row" =>"0",
       "type" => "EasyIssueQuery",
       "uuid" => page_module.uuid,
       "easy_query_type" => "EasyIssueQuery",
       "format" => "json"
      }
    }

    it 'returns correct chart data' do
      FactoryGirl.create_list :issue, 2
      get :chart, page_module.settings['chart_settings'].merge(chart_request_params)
      estimated_hours_sum = EasyIssueQuery.new.entities.sum {|i| i.estimated_hours.to_i}
      expect(chart_values.sum).to eq(estimated_hours_sum)
    end

    it 'returns no data with incorrect chart settings' do
      FactoryGirl.create_list :issue, 2

      chart_request_params['uuid'] = page_module_with_incorrect_settings.uuid

      get :chart, page_module_with_incorrect_settings.settings['chart_settings'].merge(chart_request_params)
      estimated_hours_sum = 0
      expect(chart_values.sum).to eq(estimated_hours_sum)
    end

    it 'returns correct chart data grouped_by_date_column' do
      FactoryGirl.create_list :issue, 2, :start_date => '2016-05-05'

      settings = {
        'period_start_date' => '2016-01-01',
        'period_end_date' => '2016-12-31',
        'period_zoom' => 'month',
        'period_date_period_type' => '2',
        'set_filter' => '1',
        'type' => 'EasyIssueQuery',
        'outputs' => ['chart'],
        'chart' => '1',
        'group_by' => 'start_date',
        'format' => 'json',
        'chart_settings' => {
          'primary_renderer' => 'bar',
          'axis_x_column' => 'start_date',
          'axis_y_type'   => 'sum',
          'axis_y_column' => 'estimated_hours',
          'secondary_axis_y_column' => '',
          'bar_direction' => 'vertical',
          'period_column' => 'start_date',
          'bar_limit' => '0',
          'legend_enabled' => '0',
          'legend' => {
            'location' => 'nw',
            'placement' => 'insideGrid'
          }
        }
      }

      get :chart, settings
      expect(chart_values.sum).to eq(Issue.sum(:estimated_hours))
    end

    it 'returns correct groups' do
      FactoryGirl.create :issue, :start_date => '2016-05-05', :due_date => '2016-05-05', :estimated_hours => 1
      FactoryGirl.create :issue, :start_date => '2016-05-05', :due_date => '2016-06-05', :estimated_hours => 2

      settings = {
        'period_start_date' => '2016-01-01',
        'period_end_date' => '2016-12-31',
        'period_zoom' => 'month',
        'period_date_period_type' => '2',
        'set_filter' => '1',
        'easy_query_type' => 'EasyIssueQuery',
        'outputs' => ['chart'],
        'chart' => '1',
        'group_by' => 'start_date',
        'format' => 'json',
        'chart_settings' => {
          'primary_renderer' => 'bar',
          'axis_x_column' => 'start_date',
          'axis_y_type'   => 'sum',
          'axis_y_column' => 'estimated_hours',
          'secondary_axis_y_column' => '',
          'bar_direction' => 'vertical',
          'period_column' => 'start_date',
          'bar_limit' => '0',
          'legend_enabled' => '0',
          'legend' => {
            'location' => 'nw',
            'placement' => 'insideGrid'
          },
          'additional_queries' => {
            '0' => {
              'set_filter' => '1',
              'easy_query_type' => 'EasyIssueQuery',
              'outputs' => ['chart'],
              'chart' => '1',
              'group_by' => 'due_date',
              'format' => 'json',
              'chart_settings' => {
                'primary_renderer' => 'bar',
                'axis_x_column' => 'due_date',
                'axis_y_type'   => 'sum',
                'axis_y_column' => 'estimated_hours',
                'secondary_axis_y_column' => '',
                'bar_direction' => 'vertical',
                'period_column' => 'due_date',
                'bar_limit' => '0',
                'legend_enabled' => '0',
                'legend' => {
                  'location' => 'nw',
                  'placement' => 'insideGrid'
                }
              }
            }
          }
        }
      }

      get :chart, settings
      expect(chart_groups.count).to eq(12)
    end
  end

  describe 'save query', :logged => :admin do
    render_views

    let(:project) { FactoryGirl.create(:project, :identifier => 'blabla')}

    it 'assigns project with alias' do
      get :new, :type => 'EasyIssueQuery', :project_id => project
      expect(assigns(:project)).not_to be_nil
    end
  end

  describe 'show query', :logged => :admin do
    render_views

    let(:project) { FactoryGirl.create(:project)}
    let(:project2) { FactoryGirl.create(:project, :identifier => 'blabla')}
    let(:easy_issue_query){ FactoryGirl.create(:easy_issue_query) }
    let(:easy_project_query){ FactoryGirl.create(:easy_project_query) }
    let(:project_easy_issue_query){ FactoryGirl.create(:easy_issue_query, :project => project) }
    let(:project_easy_issue_query2){ FactoryGirl.create(:easy_issue_query, :project => project2) }

    it 'project query' do
      get :show, :id => easy_project_query
      expect( response ).to redirect_to(projects_path(:query_id => easy_project_query))
    end

    it 'issue query' do
      get :show, :id => easy_issue_query
      expect( response ).to redirect_to(issues_path(:query_id => easy_issue_query))
    end

    it 'project issue query' do
      get :show, :id => project_easy_issue_query
      expect( response ).to redirect_to(project_issues_path(project, :query_id => project_easy_issue_query))
    end

    it 'project issue query with identifier' do
      get :show, :id => project_easy_issue_query2
      expect( response ).to redirect_to(project_issues_path(project2, :query_id => project_easy_issue_query2))
    end
  end

end
