class EasyReportIssuesController < ApplicationController

  menu_item :issues
  before_action :find_project, :authorize, :find_issue_statuses

  helper :issues
  include IssuesHelper
  helper :reports
  include ReportsHelper

  def index
    @project_issues_cf = IssueCustomField.visible.for_project(@project).where(:field_format => EasyExtensions.reportable_issue_cfs)

    case params[:tab]
    when 'tracker'
      tracker
    when 'priority'
      priority
    when 'assigned_to'
      assigned_to
    when 'author'
      author
    when 'fixed_version'
      fixed_version
    when 'subproject'
      subproject
    when 'category'
      category
    when /cf_(\d+)$/
      custom_field
    else
      tracker
    end
  end

  private

  def find_issue_statuses
    @statuses = IssueStatus.sorted
    @chart_statuses = params[:status_id].blank? ? IssueStatus.sorted.where(:is_closed => false) : IssueStatus.sorted.where(:id => params[:status_id])
  end

  def tracker
    @rows = @trackers = @project.trackers
    @data = @issues_by_tracker = Issue.by_tracker(@project)
    @field = 'tracker_id'
    @report_title = l(:field_tracker)
  end

  def priority
    @rows = @priorities = IssuePriority.all.reverse
    @data = @issues_by_priority = Issue.by_priority(@project)
    @field = 'priority_id'
    @report_title = l(:field_priority)
  end

  def assigned_to
    @rows = @assignees = (Setting.issue_group_assignment? ? @project.principals : @project.users).sort
    @data = @issues_by_assigned_to = Issue.by_assigned_to(@project)
    @field = 'assigned_to_id'
    @report_title = l(:field_assigned_to)
  end

  def author
    @rows = @authors = @project.users.sort
    @data = @issues_by_author = Issue.by_author(@project)
    @field = 'author_id'
    @report_title = l(:field_author)
  end

  def fixed_version
    @rows = @versions = @project.shared_versions.sort
    @data = @issues_by_version = Issue.by_version(@project)
    @field = 'fixed_version_id'
    @report_title = l(:field_version)
  end

  def subproject
    @rows = @subprojects = @project.descendants.visible
    @data = @issues_by_subproject = Issue.by_subproject(@project) || []
    @field = 'project_id'
    @report_title = l(:field_subproject)
  end

  def category
    @rows = @categories = @project.issue_categories
    @data = @issues_by_category = Issue.by_category(@project)
    @field = 'category_id'
    @report_title = l(:field_category)
  end

  def custom_field
    @issues_cf = Issue.by_custom_fields(@project)
    @field = params[:tab]
    cf_id = params[:tab].split('_').last
    @issue_custom_field = IssueCustomField.find(cf_id)

    cf_format = @issue_custom_field.format
    if cf_format.respond_to?(:possible_values_for_report)
      @rows = cf_format.possible_values_for_report(@issue_custom_field).map do |values|
        EasyReportsCfPossibleValue.new(*values)
      end
    else
      @rows = @issue_custom_field.possible_values.map{|value| EasyReportsCfPossibleValue.new(value)}
    end

    @data = Issue.by_custom_field(@issue_custom_field, @project)
    @report_title = @issue_custom_field.name
  end

end
